-- Copyright (c) 2020 Broadcom. All Rights Reserved.
-- The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
--
-- THIS SOFTWARE CONTAINS CONFIDENTIAL INFORMATION AND TRADE SECRETS OF
-- BROADCOM. USE, DISCLOSURE OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
-- EXPRESS WRITTEN PERMISSION OF BROADCOM.
--
-- The Licensed Software and Documentation are deemed to be commercial computer
-- software as defined in FAR 12.212 and subject to restricted rights as
-- defined in FAR Section 52.227-19 "Commercial Computer Software - Restricted
-- Rights" and DFARS 227.7202, Rights in "Commercial Computer Software or
-- Commercial Computer Software Documentation," as applicable, and any successor
-- regulations, whether delivered by Broadcom as on premises or hosted services.
-- Any use, modification, reproduction release, performance, display or
-- disclosure of the Licensed Software and Documentation by the U.S. Government
-- shall be solely in accordance with the terms of this Agreement.

USE [CSM_Reports]
GO
SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'spGenerateCCSEventDataForICDx') AND type in (N'P'))
BEGIN
	DROP PROCEDURE [dbo].[spGenerateCCSEventDataForICDx]
END

IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'symcICDxEventData') AND type in (N'U'))
BEGIN
	DROP TABLE [dbo].[symcICDxEventData]
END

CREATE TABLE [dbo].[symcICDxEventData]
(
	[seq_num] [int] NOT NULL,
	[policy$rule_uid] [uniqueidentifier] NULL,
	[device_uid] [uniqueidentifier] NULL,
	[device_domain] [nvarchar](4000) NULL,
	[device_name] [nvarchar](4000) NULL,
	[device_alias_name] [nvarchar](4000) NULL,
	[device_ip] [nvarchar](4000) NULL,
	[device_type] [nvarchar](100) NULL,
	[device_os_name] [nvarchar](4000) NULL,
	[policy$name] [nvarchar](400) NULL,
	[policy$rule_name] [nvarchar](max) NULL,
	[device_time] [datetime] NULL,
	[message] [varchar](max) NULL,
	[id] [int] NULL,
	[IsExceptionApplied] [bit] NULL,
	[cvssv2$risk] [decimal](5, 2) NULL,
	[severity_id] [int] NOT NULL,
	[cvssv2$confidentiality_impact_id] [int] NULL,
	[cvssv2$integrity_impact_id] [int] NULL,
	[cvssv2$availability_impact_id] [int] NULL,
	[cvssv2$access_complexity_id] [int] NULL,
	[cvssv2$attack_vector_id] [int] NULL,
	[cvssv2$authentication_id] [int] NULL
) 
GO

IF  NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'ICDx_RowNumbers') AND type in (N'SO'))
BEGIN
	CREATE SEQUENCE [dbo].[ICDx_RowNumbers] AS bigint
	START WITH 1
	INCREMENT BY 1;
END

GO

USE [CSM_Reports]
GO
/******
  Object:  StoredProcedure [dbo].[spGenerateCCSEventDataForICDx]
  Script Date: 2/12/2020 1:05:01 PM
******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

-----------------------------------------------------------------------------------
---- This script extracts the event data used by ICDx collector.
----
---- Behaviour:
----        Script will extract and transform "All" event data.
----        Most recent data is reported for asset+standard+Check combination.
----        Since extraction and transformation can sometime become time consuming,
----        the procedure once executed preserves a copy of returned results in a
----        table named "symcICDxEventData".
----        If you want to regenerate event data because you believe events would
----        have changed, then just re-execute this stored procedure. It will
----        destroy previous result-set and generate new results
---- Inputs:
----        None
---- Output:
----        Results containing detailed event information including event source,
----        source attributes, event attributes.
-----------------------------------------------------------------------------------

CREATE PROC [dbo].[spGenerateCCSEventDataForICDx]
AS
BEGIN
	SET NOCOUNT ON
	SET TRANSACTION ISOLATION LEVEL READ UNCOMMITTED

	IF OBJECT_ID('tempdb..#AssetTestSpecificEvidence') IS NOT NULL
		DROP TABLE #AssetTestSpecificEvidence

	CREATE TABLE #AssetTestSpecificEvidence
	(
		ObjectDisplayName		VARCHAR(MAX),
		ObjectTypeDisplayName	VARCHAR(MAX),
		FieldDisplayName		VARCHAR(MAX),
		FailedExpression		VARCHAR(MAX),
		RunTimeCurrentValue		VARCHAR(MAX),
		CustomMessage			VARCHAR(MAX)
	)

	IF OBJECT_ID('tempdb..#SCRWEIntermediate') IS NOT NULL
		DROP TABLE #SCRWEIntermediate

	CREATE TABLE #SCRWEIntermediate
	(
		CheckID UniqueIdentifier,
		CheckVersion VARCHAR (255),
		AssetID UniqueIdentifier,
		EvaluationDate DATETIME,
		Result TINYINT,
		ApplyException BIT,
		RiskScore DECIMAL(5,2),
		ObjectDisplayName VARCHAR(512),
		ObjectTypeDisplayName  VARCHAR(512),
		FieldDisplayName  VARCHAR(2048),
		FailedExpression  VARCHAR(2048),
		RunTimeCurrentValue VARCHAR(8000),
		CustomMessage VARCHAR(8000),
		FormattedEvidence VARCHAR(MAX)
	);

	DECLARE @CheckID UniqueIdentifier
	DECLARE @CheckVersion VARCHAR (255)
	DECLARE @AssetID UniqueIdentifier
	DECLARE @EvaluationDate DATETIME
	DECLARE @Result INT
	DECLARE @ApplyException BIT
	DECLARE @RiskScore INT
	DECLARE @EvidenceXML XML
	DECLARE @docid INT

	DECLARE @debug BIT
	SET @debug = 0
	DECLARE @tStart DATETIME;
	DECLARE @tEnd DATETIME;
	DECLARE @elapsedSeconds INT
	DECLARE @errMsg VARCHAR(1024)
	DECLARE @loopCounter INT = 0
	DECLARE @totalRowsProcessed INT = 0
	DECLARE @batchsize INT = 5000
	DECLARE @totalTStart DATETIME;
	DECLARE @precessingAllOrDeltaRows VARCHAR(32)

	IF(@debug = 1)
	BEGIN
		SET @errMsg = 'Identifying procesing mode.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT
	END

	IF OBJECT_ID('tempdb..#CursorRows') IS NOT NULL
		DROP TABLE #CursorRows

	SELECT CheckID, CheckVersion, SCR.AssetID, EvaluationDate, Result, ApplyException, RiskScore, Evidence
	INTO #CursorRows
	FROM vStandardCheckResultsWithEvidence SCR WITH (NOLOCK)
	WHERE 1 = 2

	IF  NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'symcICDxEventData') AND type in (N'U'))
	BEGIN
		SET @precessingAllOrDeltaRows = 'All';

		IF(@debug = 1)
		BEGIN
			SET @errMsg = 'Procesing mode: Process ' +@precessingAllOrDeltaRows +' rows.'
			RAISERROR (@errMsg, 0, 1) WITH NOWAIT

			SET @errMsg = 'Selecting qualifying data for processing. This step can take a long time depending on multiple factors.'
			RAISERROR (@errMsg, 0, 1) WITH NOWAIT

			SET @tStart = GETDATE();
		END

		INSERT INTO #CursorRows
		SELECT CheckID, CheckVersion, SCR.AssetID, EvaluationDate, Result, ApplyException, RiskScore, Evidence
		FROM vStandardCheckResultsWithEvidence SCR WITH (NOLOCK)
		WHERE
		SCR.isCurrentValue = 1
	END
	ELSE
	BEGIN
		SET @precessingAllOrDeltaRows = 'Only Delta';

		IF(@debug = 1)
		BEGIN
			SET @errMsg = 'Procesing mode: Process ' +@precessingAllOrDeltaRows +' rows.'
			RAISERROR (@errMsg, 0, 1) WITH NOWAIT

			SET @errMsg = 'Selecting qualifying data for processing. This step can take a long time depending on multiple factors.'
			RAISERROR (@errMsg, 0, 1) WITH NOWAIT

			SET @tStart = GETDATE();
		END

		INSERT INTO #CursorRows
		SELECT SCR.CheckID, SCR.CheckVersion, SCR.AssetID, SCR.EvaluationDate, SCR.Result, SCR.ApplyException, SCR.RiskScore, SCR.Evidence
		FROM vStandardCheckResultsWithEvidence SCR
		LEFT OUTER JOIN symcICDxEventData I ON (I.policy$rule_uid = SCR.CheckID AND I.device_uid = SCR.AssetID)
		WHERE SCR.isCurrentValue = 1
		AND SCR.EvaluationDate > ISNULL(I.[device_time], '2000-12-31')

		CREATE NONCLUSTERED INDEX nci_cursorRows_checkID_assetID ON #CursorRows(CheckID, AssetID)

		DELETE I
		FROM symcICDxEventData I
		INNER JOIN #CursorRows C ON (C.CheckID = I.policy$rule_uid AND C.AssetID = I.device_uid)
	END

	IF(@debug = 1)
	BEGIN
		SET @tEnd = GETDATE();
		SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
		SET @errMsg = 'Identifying procesing mode and selecting qualifying data for processing completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT


		SET @tStart = GETDATE();
	END

	DECLARE XMLTableCursor CURSOR FOR
	SELECT CheckID, CheckVersion, AssetID, EvaluationDate, Result, ApplyException, RiskScore, Evidence  FROM #CursorRows

	IF(@debug = 1)
	BEGIN
		SET @tEnd = GETDATE();
		SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
		SET @errMsg = 'DECLARE XMLTableCursor completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT
	END

	IF(@debug = 1)
		SET @tStart = GETDATE();

	OPEN XMLTableCursor
	FETCH NEXT FROM XMLTableCursor INTO @CheckID, @CheckVersion, @AssetID, @EvaluationDate, @Result, @ApplyException, @RiskScore, @EvidenceXML

	IF(@debug = 1)
	BEGIN
		SET @tEnd = GETDATE();
		SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
		SET @errMsg = 'OPEN XMLTableCursor completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT
	END

	IF(@debug = 1)
	BEGIN
		SET @tStart = GETDATE();
		SET @totalTStart = GetDate();
	END

	WHILE @@FETCH_STATUS = 0
	BEGIN
		---||||||||||||||||||||||||| Start: Evidence Parsing ||||||||||||||||||||||||||||
		BEGIN
		IF @EvidenceXML IS NOT NULL
		BEGIN
				BEGIN TRY
					EXEC sp_xml_preparedocument @docid OUTPUT, @EvidenceXML

					DECLARE @CustomMessage nvarchar(max);
					TRUNCATE TABLE #AssetTestSpecificEvidence;

					Select @CustomMessage = (select  CustomMessage
					from openxml(@docid, '/ev/cmsg/cm')
					with (
					CustomMessage nvarchar(max) '.'
					) customMessageTable)

					if EXISTS
					(
						-- Final select values to be returned
						Select
						source1.ObjectDisplayName,
						source1.ObjectTypeDisplayName,
						source2.FieldDisplayName,
						source1.FailedExpress as FailedExpression,
						RunTimeCurrentValue =
						(
							CASE WHEN (source1.CurrentValueList  <> '')
									THEN
									(
										Select li + ';' AS [data()]
										FROM
										OPENXML(@docid, '/ev/ers/o/er/fv/cv/li', 3)
										WITH (
										li nvarchar(max) '.' ,
										Id nvarchar(3999) '../../@id',
										MosType nvarchar(3999) '../../../../@mt' )
										where Id  = source1.Id and MosType = source1.MosType
										FOR XML PATH('')
									)
								ELSE
									source1.CurrentValue
								END
						),
						@CustomMessage as CustomMessage
						from
						openxml(@docid, '/ev/ers/o/er/fv')
						with
						(
							ObjectDisplayName nvarchar(max) '../@odn',
							ObjectTypeDisplayName nvarchar(max) '../../@mtdn',
							Id nvarchar(3999) '@id',
							MosType nvarchar(3999) '../../@mt',
							FailedExpress nvarchar(max) 'e',
							CurrentValue nvarchar(max) 'cv',
							CurrentValueList nvarchar(max) 'cv/li'
						)	source1
						Cross Apply
						openxml(@docid, '/ev/ers/o/s/f')
						with
						(
							MosType nvarchar(3999) '../../@mt',
							Id nvarchar(3999) '@id',
							FieldDisplayName nvarchar(100) '@mdn'
						) source2
						where source1.Id = source2.Id and source1.MosType = source2.MosType
					)
					BEGIN
						INSERT INTO #AssetTestSpecificEvidence
						Select
							source1.ObjectDisplayName,
							source1.ObjectTypeDisplayName,
							source2.FieldDisplayName,
							source1.FailedExpress as FailedExpression,
							RunTimeCurrentValue =
							(
								CASE  WHEN (source1.CurrentValueList  <> '')
										THEN (
												Select li + ';' AS [data()]
												FROM
												OPENXML(@docid, '/ev/ers/o/er/fv/cv/li', 3)
												WITH
												(
													li nvarchar(max) '.' ,
													Id nvarchar(3999) '../../@id',
													MosType nvarchar(3999) '../../../../@mt'
												)
												where Id  = source1.Id and MosType = source1.MosType
												FOR XML PATH('')
											)
										ELSE source1.CurrentValue
								END
							),
							@CustomMessage as CustomMessage
							from
							openxml(@docid, '/ev/ers/o/er/fv')
							with
							(
								ObjectDisplayName nvarchar(max) '../@odn',
								ObjectTypeDisplayName nvarchar(max) '../../@mtdn',
								Id nvarchar(3999) '@id',
								MosType nvarchar(3999) '../../@mt',
								FailedExpress nvarchar(max) 'e',
								CurrentValue nvarchar(max) 'cv',
								CurrentValueList nvarchar(max) 'cv/li'
							) source1
							Cross Apply
							openxml(@docid, '/ev/ers/o/s/f')
							with
							(
								MosType nvarchar(3999) '../../@mt',
								Id nvarchar(3999) '@id',
								FieldDisplayName nvarchar(100) '@mdn'
							) source2
							where source1.Id = source2.Id and source1.MosType = source2.MosType
							ORDER BY source1.ObjectTypeDisplayName, source1.ObjectDisplayName

							---||||||||||||||||||||||||| START: Evidence Formatting ||||||||||||||||||||||||||||
							BEGIN
								DECLARE @IndividualObjectDisplayName		VARCHAR(MAX)
								DECLARE @IndividualObjectTypeDisplayName	VARCHAR(MAX)
								DECLARE @IndividualFieldDisplayName			VARCHAR(MAX)
								DECLARE @IndividualFailedExpression			VARCHAR(MAX)
								DECLARE @IndividualRunTimeCurrentValue		VARCHAR(MAX)
								DECLARE @IndividualCustomMessage			VARCHAR(MAX)
								DECLARE @FormattedEvidence					VARCHAR(MAX)
								SET @FormattedEvidence = ''

								DECLARE XMLIndividualEvidenceCursor CURSOR FOR
								SELECT
									ObjectDisplayName,
									ObjectTypeDisplayName,
									FieldDisplayName,
									FailedExpression,
									RunTimeCurrentValue  ,
									CustomMessage
									FROM #AssetTestSpecificEvidence

								OPEN XMLIndividualEvidenceCursor
								FETCH NEXT FROM XMLIndividualEvidenceCursor INTO
								@IndividualObjectDisplayName,
								@IndividualObjectTypeDisplayName,
								@IndividualFieldDisplayName,
								@IndividualFailedExpression,
								@IndividualRunTimeCurrentValue,
								@IndividualCustomMessage

								WHILE @@FETCH_STATUS = 0
								BEGIN

									IF @FormattedEvidence = ''
										SET @FormattedEvidence = @FormattedEvidence + CHAR(13)  + CHAR(13)

									SET @FormattedEvidence = @FormattedEvidence +'Object Name   : ' +@IndividualObjectDisplayName		+ CHAR(13)
									SET @FormattedEvidence = @FormattedEvidence +'Object Type   : ' +@IndividualObjectTypeDisplayName	+ CHAR(13)
									SET @FormattedEvidence = @FormattedEvidence +'Expression    : ' +@IndividualFailedExpression		+ CHAR(13)
									SET @FormattedEvidence = @FormattedEvidence +'Current Value : ' +@IndividualRunTimeCurrentValue		+ CHAR(13)

									IF LEN(@FormattedEvidence) > 64535
									BEGIN
										SET @FormattedEvidence = LEFT(@FormattedEvidence, 64499) + ' <<... data truncated to 64KB limit>>'
										BREAK;
									END

									FETCH NEXT FROM XMLIndividualEvidenceCursor INTO
									@IndividualObjectDisplayName,
									@IndividualObjectTypeDisplayName,
									@IndividualFieldDisplayName,
									@IndividualFailedExpression,
									@IndividualRunTimeCurrentValue,
									@IndividualCustomMessage
								END
								CLOSE XMLIndividualEvidenceCursor
								DEALLOCATE XMLIndividualEvidenceCursor

								INSERT INTO #SCRWEIntermediate
								SELECT
								@CheckID,
								@CheckVersion,
								@AssetID,
								@EvaluationDate,
								@Result,
								@ApplyException,
								@RiskScore,
								NULL as ObjectDisplayName,
								NULL as ObjectTypeDisplayName,
								NULL as FieldDisplayName,
								NULL as FailedExpression,
								NULL as RunTimeCurrentValue,
								NULL as CustomMessage,
								@FormattedEvidence as FormattedEvidence
							END
							---||||||||||||||||||||||||| END: Evidence Formatting ||||||||||||||||||||||||||||||
					END
					ELSE
					BEGIN
						INSERT INTO #SCRWEIntermediate
						select
						@CheckID,
						@CheckVersion,
						@AssetID,
						@EvaluationDate,
						@Result,
						@ApplyException,
						@RiskScore,
						NULL as ObjectDisplayName,
						NULL as ObjectTypeDisplayName,
						NULL as FieldDisplayName,
						NULL as FailedExpression,
						NULL as RunTimeCurrentValue,
						NULL as CustomMessage,
						NULL as FormattedEvidence
					END
				END TRY
				BEGIN CATCH
					INSERT INTO #SCRWEIntermediate
					SELECT
					@CheckID,
					@CheckVersion,
					@AssetID,
					@EvaluationDate,
					@Result,
					@ApplyException,
					@RiskScore,
					NULL as ObjectDisplayName,
					NULL as ObjectTypeDisplayName,
					NULL as FieldDisplayName,
					NULL as FailedExpression,
					NULL as RunTimeCurrentValue,
					NULL as CustomMessage,
					NULL as FormattedEvidence
				END CATCH
				EXEC sp_xml_removedocument @docid
		END
		ELSE
		BEGIN
			INSERT INTO #SCRWEIntermediate
			SELECT
			@CheckID,
			@CheckVersion,
			@AssetID,
			@EvaluationDate,
			@Result,
			@ApplyException,
			@RiskScore,
			NULL as ObjectDisplayName,
			NULL as ObjectTypeDisplayName,
			NULL as FieldDisplayName,
			NULL as FailedExpression,
			NULL as RunTimeCurrentValue,
			NULL as CustomMessage,
			NULL as FormattedEvidence
		END
		END
		---||||||||||||||||||||||||| END: Evidence Parsing ||||||||||||||||||||||||||||
		IF(@debug = 1)
		BEGIN
			SET @loopCounter = @loopCounter + 1
			SET @totalRowsProcessed = @totalRowsProcessed + 1
			IF(@loopCounter % @batchsize = 0)
			BEGIN
				SET @tEnd = GETDATE();
				SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
				SET @errMsg = 'Batch of ' +CAST (@batchsize AS VARCHAr(8))+' records completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds. Total processed so far: ' +CAST(@totalRowsProcessed AS VARCHAR(16)) +'. Average records per hour: ' + CAST(3600000/DATEDIFF(millisecond,@totalTStart,@tEnd)*@totalRowsProcessed AS VARCHAR(32));
				RAISERROR (@errMsg, 0, 1) WITH NOWAIT
				SET @tStart = GETDATE();
			END
		END


		FETCH NEXT FROM XMLTableCursor INTO @CheckID, @CheckVersion, @AssetID, @EvaluationDate, @Result, @ApplyException, @RiskScore, @EvidenceXML
	END
	CLOSE XMLTableCursor
	DEALLOCATE XMLTableCursor

	IF(@debug = 1)
		SET @tStart = GETDATE();

	CREATE CLUSTERED INDEX ci_#SCRWEIntermediate_cid_cv_aid ON #SCRWEIntermediate(CheckID, CheckVersion, AssetID)

	IF(@debug = 1)
	BEGIN
		SET @tEnd = GETDATE();
		SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
		SET @errMsg = 'Creating clustered index on #SCRWEIntermediate completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT
	END

	IF(@debug = 1)
	BEGIN
		SET @tStart = GETDATE();
		RAISERROR ('Loading results in "symcICDxEventData" table.', 0, 1) WITH NOWAIT
	END

	INSERT INTO [dbo].[symcICDxEventData]
	SELECT
	NEXT VALUE FOR ICDx_RowNumbers			[seq_num],
	s.CheckID								[policy$rule_uid],
	s.AssetID								[device_uid],
	CASE AssetTypeName
	WHEN 'Windows Machine' THEN
		CASE WHEN CHARINDEX('\', AssetName)  > 0 AND SUBSTRING(AssetName, 1, CHARINDEX('\', AssetName) - 1) <> Hostname THEN SUBSTRING(AssetName, 1, CHARINDEX('\', AssetName) - 1)
		END
	END										[device_domain],

	CASE AssetTypeName
	WHEN 'Oracle Database' THEN
		CASE CHARINDEX('.', AssetName)
			WHEN 0 THEN AssetName
			ELSE SUBSTRING(AssetName, CHARINDEX('.', AssetName) + 1, LEN(AssetName))
		END
	ELSE ISNULL(HostName, AssetName)
	END										[device_name],
	vA.AssetName							[device_alias_name],
	CASE WHEN CHARINDEX(',', vA.IPAddress) > 0	THEN SUBSTRING(IPAddress, 1, CHARINDEX(',', IPAddress) - 1)
		 ELSE vA.IPAddress
	END					 					[device_ip],
	vA.AssetTypeName						[device_type],
	CASE 
		WHEN vA.AssetTypeName = 'Windows Machine'
		THEN ISNULL((SELECT Value FROM vAssetAttributesLookup WITH (NOLOCK) WHERE Name = 'OSVersion' AND AssetID = vA.AssetID), vA.AssetTypeName)

		WHEN vA.AssetTypeName = 'Unix Machine'
		THEN ISNULL((SELECT Value FROM vAssetAttributesLookup WITH (NOLOCK) WHERE Name = 'OSDistribution' AND AssetID = vA.AssetID), vA.AssetTypeName)

		ELSE vA.AssetTypeName
	END										[device_os_name],
	std.StandardName						[policy$name],
	chk.[Name]								[policy$rule_name],
	s.EvaluationDate						[device_time],
	s.FormattedEvidence						[message],
	CASE s.[Result]
		When 1 THEN 1 /*CCS Pass	= 1, ICDx Pass	  = 1*/
		When 2 Then 2 /*CCS Fail	= 2, ICDx Fail	  = 2*/
		When 3 Then 0 /*CCS Unknown = 3, ICDx Unknown = 0*/
		When 4 Then 4 /*CCS N/A		= 4, ICDx N/A	  = 4*/
		When 5 Then 3 /*CCS Error	= 5, ICDx Error   = 3*/
	END										[id],
	s.ApplyException IsExceptionApplied,
	s.RiskScore								[cvssv2$risk],
	CASE WHEN Result IN (1, 3, 4, 5) THEN 1				/* If a check status is anything but Fail then ICDx Severity is Information (1)*/
		 WHEN s.RiskScore >= 9.0 THEN 5					/* CCS Risk score 9 and 10 = ICDx Critical (5) */
		 WHEN s.RiskScore >= 6.0 THEN 4					/* CCS Risk score 6, 7 and 8 = ICDx Major (4) */
		 WHEN s.RiskScore >= 4.0 THEN 3					/* CCS Risk score 4 and 5 = ICDx Minor (3) */
		 ELSE 2											/* CCS Risk score 1,2 and 3 or check has Failed with RiskScore -1 = ICDx Warning (2) */
	END										[severity_id],
	CAST(chk.ConfidentialityRating AS INT)	[cvssv2$confidentiality_impact_id],
	CAST(chk.IntegrityRating AS INT)		[cvssv2$integrity_impact_id],
	CAST(chk.AvailabilityRating AS INT)		[cvssv2$availability_impact_id],
	CAST(chk.AccessComplexity AS INT)		[cvssv2$access_complexity_id],
	CAST(chk.AccessVector AS INT)			[cvssv2$attack_vector_id],
	CAST(chk.Authentication AS INT)			[cvssv2$authentication_id]
	FROM #SCRWEIntermediate s
	INNER JOIN [CSM_Reports].[dbo].[vCheck] chk ON (s.CheckID = chk.TestID and s.[checkversion] = chk.[Testversion])
	INNER JOIN [CSM_Reports].[dbo].[vStandardChecks] chkStd ON (chk.Testid = chkStd.CheckID AND chk.testVersion = chkStd.CheckVersion)
	INNER JOIN [CSM_Reports].[dbo].[vStandard] std ON (chkStd.StandardID = std.StandardID AND chkStd.StandardVersion = std.StandardVersion)
	INNER JOIN [CSM_Reports].[dbo].[vAsset] vA ON vA.AssetID = s.AssetID
	WHERE va.isCurrentVersion = 1

	IF(@debug = 1)
	BEGIN
		SET @tEnd = GETDATE();
		SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
		SET @errMsg = 'Loading results in "symcICDxEventData" table completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT
	END


	IF(@debug = 1)
	BEGIN
		SET @tStart = GETDATE();
	END

	BEGIN TRY
		IF EXISTS (SELECT *  FROM sys.indexes  WHERE name='ci_symcICDxEventData_seqnum' AND object_id = OBJECT_ID('[dbo].[symcICDxEventData]'))
		BEGIN
			DROP INDEX [Index_Name] ON [SchmaName].[TableName];
		END
		CREATE CLUSTERED INDEX ci_symcICDxEventData_seqnum ON symcICDxEventData(seq_num)
	END TRY
	BEGIN CATCH
		--Ignore the error, if any
	END CATCH

	IF(@debug = 1)
	BEGIN
		SET @tEnd = GETDATE();
		SELECT @elapsedSeconds = DATEDIFF(second,@tStart,@tEnd);
		SET @errMsg = 'Creating clustered index on symcICDxEventData completed in ' +CAST(@elapsedSeconds AS VARCHAr(16)) +' seconds.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT

		SET @errMsg = 'Script completed successfully. Processed "' +@precessingAllOrDeltaRows +'" rows.'
		RAISERROR (@errMsg, 0, 1) WITH NOWAIT
	END

END
GO