#!/usr/bin/env bash
# This script creates an Elasticsearch index template for Elasticsearch 6.
# https://www.elastic.co/guide/en/elasticsearch/reference/6.8/indices-templates.html

dry_run=
index='icdx_events-*'
shards=5
replicas=1
host=localhost
port=9200
use_ssl=
ssl_insecure=
template_name=icdx_events_template_v1
user_password=
prompt_user_password=
debug=

OPTIND=1

function usage() {
  echo "Create Elasticsearch index template."
  echo "Usage: $0 [options]"
  echo " -h                 Show this help"
  echo " -d                 Show the index template"
  echo " -x                 Show the index template and connection options"
  echo " -i INDEX_PATTERN   Index pattern; default: ${index}"
  echo " -s SHARDS          Number of shards for each index; default: ${shards}"
  echo " -r REPLICAS        Number of replicas for each index; default: ${replicas}"
  echo " -n HOSTNAME        Elasticsearch node hostname; default: ${host}"
  echo " -p PORT            Elasticsearch node HTTP port; default: ${port}"
  echo " -t                 Use SSL/TLS"
  echo " -k                 Continue when SSL connection is considered insecure"
  echo " -u USER[:PASSWORD] User name and password; no default"
  echo " -v                 Prompt for user name and password"
  echo " -e TEMPLATE_NAME   Index template name; default:"
  echo "                        ${template_name}"
  echo
  echo "Notes:"
  echo " * Index pattern must match Elasticsearch Forwarder."
  echo " * 5 shards is good for multi-node clusters, while 1 is good for single-node."
  echo " * 1 replica is good for multi-node clusters, while 0 is good for single-node."
}

while getopts ":hdi:s:r:m:n:p:tku:ve:x" opt; do
    case "$opt" in
        h)
            usage
            exit 0
            ;;
        d)
            dry_run="on"
            ;;
        i)
            index=${OPTARG}
            ;;
        s)
            shards=${OPTARG}
            ;;
        r)
            replicas=${OPTARG}
            ;;
        n)
            host=${OPTARG}
            ;;
        p)
            port=${OPTARG}
            ;;
        t)
            use_ssl="on"
            ;;
        k)
            ssl_insecure="on"
            ;;
        u)
            user_password=${OPTARG}
            ;;
        v)
            prompt_user_password="on"
            ;;
        e)
            template_name=${OPTARG}
            ;;
        x)
            debug="on"
            ;;
        \?)
            echo "Invalid option: -${OPTARG}" >&2
            usage
            exit 1
            ;;
    esac
done

read -r -d '' template <<end-of-file
{
  "index_patterns": "${index}",
  "order": 0,
  "settings": {
    "number_of_shards": ${shards},
    "number_of_replicas": ${replicas},
    "analysis": {
      "analyzer": {
        "lowercase_only": {
          "type": "custom",
          "tokenizer": "keyword",
          "filter": ["lowercase"]
        }
      }
    }
  },
  "mappings": {
    "_doc": {
      "dynamic": false,
      "date_detection": false,
      "numeric_detection": false,

      "properties": {
        "log_time": {"type": "date"},
        "time": {"type": "date"},

        "type_id": {"type": "text", "analyzer": "lowercase_only"},
        "id": {"type": "text", "analyzer": "lowercase_only"},
        "event_id": {"type": "integer"},

        "category_id": {"type": "text", "analyzer": "lowercase_only"},
        "severity_id": {"type": "text", "analyzer": "lowercase_only"},

        "collector_name": {"type": "text", "analyzer": "lowercase_only"},
        "collector_uid": {"type": "text", "analyzer": "lowercase_only"},

        "device_ip": {"type": "text", "analyzer": "lowercase_only"},
        "device_name": {"type": "text", "analyzer": "lowercase_only"},
        "device_os_name": {"type": "text", "analyzer": "lowercase_only"},

        "feature_name": {"type": "text", "analyzer": "lowercase_only"},
        "feature_ver": {"type": "text", "analyzer": "lowercase_only"},

        "product_name": {"type": "text", "analyzer": "lowercase_only"},
        "product_ver": {"type": "text", "analyzer": "lowercase_only"},

        "user_name": {"type": "text", "analyzer": "lowercase_only"}
      }
    }
  }
}
end-of-file

if [ ${dry_run} ] ; then
  echo "${template}"
  exit 0
fi

if [ ${prompt_user_password} ] ; then
  read -r -p "User name and password (user or user:password) [${user_password}]: " user_password
fi

options=("-H" "Accept: application/json" "-H" "Content-Type: application/json")

if [ ${use_ssl} ] ; then
  scheme="https"
  if [ ${ssl_insecure} ] ; then
    options+=("-k")
  fi
else
  scheme="http"
fi

if [ "${user_password}" ] ; then
  options+=("-u" "${user_password}")
fi

url=${scheme}://${host}:${port}/_template/${template_name}

if [ ${debug} ] ; then
  echo "Options:" "${options[@]}"
  echo "URL: ${url}"
  echo "Index template:"
  echo "${template}"
  exit 0
fi

# Delete existing template if it exists
if curl "${options[@]}" --head --fail -o /dev/null --silent "${url}" ; then
  echo "Removing existing index template: ${template_name}"
  curl "${options[@]}" -XDELETE "${url}"
  echo
fi

# create template
echo "Creating index template: ${template_name}"
curl "${options[@]}" "${url}" --data-binary "${template}"
echo
