#!/bin/bash

cleanup()
{
	rm -rf "${tempDir}"
}

main()
{
	verifyRoot
	
	#process command line
	backupMode=${1:-'undefined'}
	restoreType=${2:-'undefined'}
	backupLocation=${2:-'undefined'}
	restoreLocation=${3:-'undefined'}
	optionalFlag=${4:-'undefined'}

	currentDir=`pwd`

	tempDir=`mktemp -d`
	chown icdx ${tempDir} > /dev/null 2>&1

	trap cleanup EXIT

  if [ $# -eq 0 ]; then
    printHelp
    exit 1
  fi

	if [ ${backupMode} == "backup" ]; then
		performBackup
	elif [ ${backupMode} == "restore" ]; then
		performRestore
	fi

	echo "Unknown backup mode ${backupMode}."
	printHelp
	exit 1
}

verifyRoot()
{
	if [ "${EUID}" -ne 0 ]
	  then echo "Run as root user."
	  exit
	fi
}

performRestore()
{
	if [ ${restoreType} == 'keys' ]; then
		performKeysRestore
		exit 0
	elif [ ${restoreType} == 'config' ]; then
		performConfigRestore
		exit 0
	fi

	echo "Unknown restore type ${restoreType}."
	printHelp
	exit 1
}

performKeysBackup()
{
	if [ -d "/etc/icdx" ]; then
		echo "Backing up encryption keys."
		tar czvf ${tempDir}/icdx-keys.tar.gz /etc/icdx --absolute-names 1>/dev/null
		if [ $? -eq 0 ]; then
			echo "Successfully created 'icdx-keys.tar.gz'."
		else
			echo "Failed to back up encryption keys."
			exit 1
		fi
	else
		echo "Encryption keys do not exist."
		exit 1
	fi
}

getInstalledVersion()
{
	tail -n 5 ${SYMC_HOME}/portal_dx/icdx.json |grep version |awk '{printf $NF;}'
}

performBackup()
{
	#check if SYMC_HOME is set
	if [[ -z ${SYMC_HOME} && -z ${SYMC_DATA} ]]; then
		echo "SYMC_HOME and SYMC_DATA are not set. Exiting."
		exit 1
	fi

	performKeysBackup
	backupAppsConfigurations
	backupRepoConfigurations
	getInstalledVersion > ${tempDir}/version

	cd ${tempDir}

	if [ ! -z ${backupLocation+x} ] && [ ${backupLocation} != "undefined" ]; then
		if [ ! -d ${backupLocation} ]; then
			echo "Created backup location: ${backupLocation}"
			mkdir -p ${backupLocation}
			chown icdx ${backupLocation}
		fi
		finalBackupLocation=${backupLocation}
	elif [ ! -d ${SYMC_DATA}/backup ]; then
		echo "Created backup location: ${SYMC_DATA}/backup"
		mkdir -p ${SYMC_DATA}/backup
		chown icdx ${SYMC_DATA}/backup
		finalBackupLocation=${SYMC_DATA}/backup
	else
		finalBackupLocation=${SYMC_DATA}/backup
	fi

	currentDate=`date -u +"%FT%H%M%SZ"`
	tar czvf ${finalBackupLocation}/icdx-config-${currentDate}.tar.gz * 1>/dev/null

	echo "Configuration backup complete."

	exit 0
}

extractConfig()
{
	if [ ${restoreLocation} == "undefined" ] || [ ! -f ${restoreLocation} ]; then
		echo "Restore file is not specified or does not exist."
		printHelp
		exit 1
	fi

	tar xzf ${restoreLocation} --directory ${tempDir} 1>/dev/null
	if [ $? -eq 0 ]; then
		echo "Successfully extracted restore file."
	else
		echo "Failed to extract restore file."
		exit 1;
	fi
}

validateVersion()
{
	#compare backuped and installed versions
	installedVersion=`getInstalledVersion`
	backupVersion=`cat ${tempDir}/version`

	if [[ ${installedVersion} != ${backupVersion} && ${optionalFlag} != '--ignoreBackupVersion' ]]; then
	        echo "ICDx version: $installedVersion"
                echo "Backup version: $backupVersion"
		echo "Backup version does not match ICDx version. Exiting."
		exit 1
	fi
}

performKeysRestore()
{
	echo "Restoring ICDx encryption keys."
	extractConfig

	tar xzf ${tempDir}/icdx-keys.tar.gz --absolute-names 1>/dev/null
	if [ $? -eq 0 ]; then
		echo "Encryption keys successfully restored to /etc/icdx."
	else
		echo "Failed to restore encryption keys to /etc/icdx."
	fi
	exit 0
}

performConfigRestore()
{
	echo "Restoring ICDx configuration."
	#source environment variables
	set -o allexport; source /etc/environment; set +o allexport

	#check if SYMC_HOME is set
	if [[ -z ${SYMC_HOME} ]]; then
		echo "SYMC_HOME is not set. Exiting."
		exit 1
	fi

	extractConfig
	validateVersion

	restoreAppsConfigurations
	restoreRepoConfigurations

	exit 0
}

backupAppsConfigurations()
{
	backupConfig collector
	backupConfig forwarder
}

backupConfig()
{
	directory=${1}
	echo "Backing up configuration: '${SYMC_HOME}/${directory}'."

	cd ${SYMC_HOME}/repo/${directory}

	#first find looks up all the .json configurations which are practically 9d7aa940-5c9b-11eb-ef7c-000000000005.json
	#in the'repo' direcotry and pipes them to cut to remove the extensions
	#then the UID is passed xargs that executes another find in the target 'apps' directory which passed as argument to the function
	#tar everything except 'home', '.metadata' and directories to avoid duplicates
	find . -name \*.json -printf "%f\n"| cut -f 1 -d '.'| xargs -I % bash -c  "cd ${SYMC_HOME}/apps/${directory}; find . ! -type d | grep % | xargs tar -rvf ${tempDir}/icdx-${directory}.tar" 1>/dev/null
}

restoreAppsConfigurations()
{
	retstoreConfig collector
	retstoreConfig forwarder
}

retstoreConfig()
{
	directory=${1}
	if [[ -a "${tempDir}/icdx-${directory}.tar" ]]; then
		echo "Restoring configuration: ${SYMC_HOME}/apps/${directory}"
		su icdx -s /bin/bash -c  "tar -xvf \"${tempDir}/icdx-${directory}.tar\" --directory ${SYMC_HOME}/apps/${directory} 1>/dev/null"
	fi
}

backupRepoConfigurations()
{
	echo "Backing up configuration: '${SYMC_HOME}/repo'"
	cd ${SYMC_HOME}/repo
	find . -name \*.json -exec tar -rvf ${tempDir}/icdx-repo.tar '{}' \; 1>/dev/null
}

restoreRepoConfigurations()
{
	if [[ -a "${tempDir}/icdx-repo.tar" ]]; then
		echo "Restoring configuration: ${SYMC_HOME}/repo"
		su icdx -s /bin/bash -c "tar -xvf '${tempDir}/icdx-repo.tar' --directory ${SYMC_HOME}/repo 1>/dev/null"
	fi
}

printHelp()
{
	echo -e "${help}"
}

help=$(cat <<EOF

Usage: ./icdx_recovery.sh  [ backup [ path ]  | restore [ keys | config ] restorefile ]

Arguments:
  backup        Create the backup file at $SYMC_DATA/backup (default) or the specified path.         
  path          Create the backup file at the specified path.
  restore       Restore keys or config from the restorefile (e.g., icdx-config-<datetime>.tar.gz).
  keys          Restore keys only.
  config        Restore configuration only.
  restorefile   Full path to the restorefile.

For complete documentation, see "Planning for ICDx Disaster Recovery" in the online help.

EOF
)

#call main with command line params
main "$@"
