#!/usr/bin/env bash

#
# ICDx pre-install script for Red Hat and CentOS installations.
# The commands here will work for a default OS installation. You may need to
# change the SELinux and firewall commands to suit your specific needs.
#

if [ ! -f /etc/redhat-release ]; then
    echo "This tool is only intended for Red Hat based Linux distributions"
    exit 1
fi

if [[ "$EUID" -ne 0 ]]; then
    echo "Please run as root"
    exit 1
fi

set -e  # exit on error

echo
if selinuxenabled > /dev/null 2>&1 ; then
    echo "SELinux is enabled - configuring..."
    # SELinux is installed so we will configure it for ICDx
    # (regardless of whether SELinux is currently enabled)

    echo
    version=$( cat /etc/redhat-release | grep -oP "[0-9]+" | head -1 )
    if [ "${version}" = "8" ]; then
        echo "Installing policycoreutils-python-utils for semanage command"
        yum install -y policycoreutils-python-utils
    else
        echo "Installing policycoreutils-python for semanage command"
        yum install -y policycoreutils-python
    fi

    echo
    echo "Allowing Nginx to redirect AMQPS (port 5671) to local AMQP (port 5672)"
    # semanage port fails if port already added, thus the || true bit
    semanage port -a -t http_port_t -p tcp 5671 || true

    echo
    echo "Allowing Nginx to connect to network"
    # The -P (persistent) option causes an attempt to write a policy file.
    # This may fail due to policy restrictions, including no default policies.
    # In some cases, the install and ICDx will work anyway, so we will tolerate the failure.
    if ! setsebool -P httpd_can_network_connect 1 ; then
        echo "Failed to set permanent SELinux policy:"
        echo "    setsebool -P httpd_can_network_connect 1"
        echo "If this causes trouble, double-check your SELinux policies."
    fi
else
    echo "SELinux is not enabled"
fi

# Test if firewalld is enabled by checking its status
# (This will be true for default installations)
echo
if systemctl status firewalld.service > /dev/null 2>&1 ; then
    echo "Firewall service (firewalld) is enabled - configuring..."

    # Open HTTP (80)
    # If installer's -http options is used, this is required for HTTP access
    # If -http is not used, this allows browser redirection from HTTP to HTTPS
    # Do not enable if only HTTPS access is desired
    echo
    echo "Opening HTTP (port 80)"
    firewall-cmd --zone=public --add-port=80/tcp --permanent

    # Open HTTPS (443)
    # Web access is always possible over the HTTPS port
    echo
    echo "Opening HTTPS (port 443)"
    firewall-cmd --zone=public --add-port=443/tcp --permanent

    # Open AMQP (5672) - the plain text (insecure) port for RabbitMQ
    # For ICDx chaining receivers, RabbitMQ collectors, and RabbitMQ forwarders
    # Do not enable if only secure AMQPS access is desired
    # firewall-cmd --zone=public --add-port=5672/tcp --permanent

    # Open AMQPS (5671) - AMQP over SSL, the secure port for RabbitMQ
    # For ICDx chaining receivers, RabbitMQ collectors, and RabbitMQ forwarders
    echo
    echo "Opening AMQPS (port 5671)"
    firewall-cmd --zone=public --add-port=5671/tcp --permanent

    # If the Symantec Secure Web Gateway (ProxySG) collector is being, used the
    # listening port must be opened
    # echo "Open AMQP (port 5672)"
    # firewall-cmd --zone=public --add-port=9000/tcp --permanent

    # Other listening ports needed to be added in a similar way

    # Take effect before reboot
    echo
    echo "Reloading firewall so new rules take effect immediately"
    firewall-cmd --reload
else
    echo "Firewall service (firewalld) is not enabled"
fi
